// Simple Common JavaScript - Fixed Version

let socket;
let currentRedisInterval = 2;
let currentDispatcherInterval = 5;
let isRefreshing = false;

function updateConnectionStatus(statusId, textId, connected, message) {
    const statusElement = document.getElementById(statusId);
    const textElement = document.getElementById(textId);
    
    if (statusElement) {
        statusElement.className = connected ? 'status-dot online' : 'status-dot offline';
    }
    if (textElement) {
        textElement.textContent = message;
    }
}

function animateNumber(elementId, targetValue) {
    const element = document.getElementById(elementId);
    if (!element) return;
    
    const currentValue = parseInt(element.textContent) || 0;
    if (currentValue !== targetValue) {
        element.textContent = targetValue;
    }
}

function formatPhoneNumber(number) {
    if (!number || number === 'N/A') return 'N/A';
    if (number.length > 15) return number.substring(0, 15) + '...';
    return number;
}

function getCallTypeClass(callType) {
    switch(callType?.toLowerCase()) {
        case 'ibd': return 'call-type-ibd';
        case 'obd': return 'call-type-obd';
        case 'incoming': return 'call-type-incoming';
        default: return '';
    }
}

function getStatusClass(status) {
    switch(status?.toLowerCase()) {
        case 'active': return 'status-active';
        case 'checking': return 'status-checking';
        case 'inactive': return 'status-inactive';
        case 'disabled': return 'status-disabled';
        case 'unknown': return 'status-unknown';
        default: return 'status-other';
    }
}

function getLatencyClass(latency) {
    if (latency < 50) return 'latency-excellent';
    if (latency < 100) return 'latency-good';
    if (latency < 200) return 'latency-fair';
    return 'latency-poor';
}

function requestUpdate() {
    if (isRefreshing) return;
    
    console.log('Manual refresh requested');
    isRefreshing = true;
    
    const spinner = document.getElementById('refreshSpinner');
    const text = document.getElementById('refreshText');
    if (spinner) spinner.style.display = 'inline-block';
    if (text) text.textContent = 'Refreshing...';
    
    if (socket && socket.connected) {
        socket.emit('request_update');
    } else {
        // Fallback to API
        fetch('/api/calls')
            .then(response => response.json())
            .then(data => {
                console.log('Manual refresh completed via API');
                if (typeof window.updateCallData === 'function') {
                    window.updateCallData(data);
                }
            })
            .catch(error => {
                console.log(`Manual refresh error: ${error}`);
            });
    }
    
    setTimeout(() => {
        isRefreshing = false;
        if (spinner) spinner.style.display = 'none';
        if (text) text.textContent = 'Refresh';
    }, 1000);
}

// Initialize Socket.IO connection
function initializeSocket() {
    console.log('Initializing Socket.IO connection...');
    socket = io();
    
    socket.on('connect', function() {
        console.log('✓ Connected to server');
        updateConnectionStatus('connectionStatus', 'statusText', true, 'Call System Online');
    });

    socket.on('disconnect', function() {
        console.log('✗ Disconnected from server');
        updateConnectionStatus('connectionStatus', 'statusText', false, 'Connection Lost - Reconnecting...');
    });

    socket.on('redis_connection_status', function(data) {
        console.log(`Redis connection: ${data.connected ? 'Connected' : 'Disconnected'}`);
        if (data.connected) {
            updateConnectionStatus('connectionStatus', 'statusText', true, 'Call System Online');
        } else {
            updateConnectionStatus('connectionStatus', 'statusText', false, 'Call System Offline');
        }
    });

    socket.on('dispatcher_connection_status', function(data) {
        console.log(`Dispatcher connection: ${data.connected ? 'Connected' : 'Disconnected'}`);
        if (data.connected) {
            updateConnectionStatus('dispatcherStatus', 'dispatcherStatusText', true, 'Dispatchers Online');
        } else {
            updateConnectionStatus('dispatcherStatus', 'dispatcherStatusText', false, 'Dispatchers Offline');
        }
    });

    socket.on('call_data_update', function(data) {
        console.log(`Call data update: ${data.total_calls} calls`);
        if (typeof window.updateCallData === 'function') {
            window.updateCallData(data);
        }
        
        if (data.redis_interval) {
            currentRedisInterval = data.redis_interval;
        }
    });

    socket.on('dispatcher_data_update', function(data) {
        console.log(`Dispatcher data update: ${data.total_dispatchers} dispatchers`);
        if (typeof window.updateDispatcherData === 'function') {
            window.updateDispatcherData(data);
        }
        
        if (data.dispatcher_interval) {
            currentDispatcherInterval = data.dispatcher_interval;
        }
    });

    socket.on('interval_updated', function(data) {
        console.log(`Interval updated: ${data.type} = ${data.interval}s`);
        if (data.type === 'redis') {
            currentRedisInterval = data.interval;
            const select = document.getElementById('redisRefreshSelect');
            if (select && select.value != data.interval) {
                select.value = data.interval;
            }
        } else if (data.type === 'dispatcher') {
            currentDispatcherInterval = data.interval;
            const select = document.getElementById('dispatcherRefreshSelect');
            if (select && select.value != data.interval) {
                select.value = data.interval;
            }
        }
    });
}

// Initialize everything when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, initializing application...');
    
    initializeSocket();
    
    // Set up refresh interval selectors
    const redisSelect = document.getElementById('redisRefreshSelect');
    const dispatcherSelect = document.getElementById('dispatcherRefreshSelect');
    
    if (redisSelect) {
        redisSelect.addEventListener('change', function() {
            const interval = parseInt(this.value);
            console.log(`Redis interval changed to ${interval}s`);
            
            if (socket && socket.connected) {
                socket.emit('set_redis_interval', {interval: interval});
            } else {
                fetch(`/api/set_redis_interval/${interval}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            currentRedisInterval = interval;
                            console.log(`Redis interval updated to ${interval}s`);
                        }
                    })
                    .catch(error => {
                        console.log(`Error updating interval: ${error}`);
                    });
            }
        });
    }
    
    if (dispatcherSelect) {
        dispatcherSelect.addEventListener('change', function() {
            const interval = parseInt(this.value);
            console.log(`Dispatcher interval changed to ${interval}s`);
            
            if (socket && socket.connected) {
                socket.emit('set_dispatcher_interval', {interval: interval});
            } else {
                fetch(`/api/set_dispatcher_interval/${interval}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            currentDispatcherInterval = interval;
                            console.log(`Dispatcher interval updated to ${interval}s`);
                        }
                    })
                    .catch(error => {
                        console.log(`Error updating interval: ${error}`);
                    });
            }
        });
    }
    
    console.log('Application initialization completed');
});
