#!/usr/bin/env python3
"""
Kamailio Live Calls Monitor - Production Version
Real Redis and Kamailio integration without sample data
"""

import subprocess
import json
import re
import time
import base64
from datetime import datetime, timedelta
from threading import Thread, Lock
from flask import Flask, render_template, jsonify, request, session, redirect, url_for, flash
from flask_socketio import SocketIO, emit, disconnect
from werkzeug.security import check_password_hash, generate_password_hash
import logging
import os
from functools import wraps
import redis
import pymysql
import socket
import struct

MYSQL_CONFIG = {
    'host': 'localhost',
    'port': 3636,
    'user': 'root',
    'password': 'Suraj@123',
    'database': 'kamailio',
    'charset': 'utf8mb4'
}

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

app = Flask(__name__)
app.config['SECRET_KEY'] = 'sarv-kamailio-secret-key-2024'
app.config['PERMANENT_SESSION_LIFETIME'] = timedelta(minutes=15)
socketio = SocketIO(app, cors_allowed_origins="*")

# Redis Configuration
REDIS_HOST = '127.0.0.1'
REDIS_PORT = 4321
REDIS_KEY = 'TotalCalls'

# Authentication credentials
USERNAME = 'sarvkamailio'
PASSWORD_HASH = generate_password_hash('sarv@kamailio#123')

def login_required(f):
    """Decorator to require login for routes"""
    @wraps(f)
    def decorated_function(*args, **kwargs):
        if 'logged_in' not in session or not session['logged_in']:
            return redirect(url_for('login'))
        
        if 'last_activity' in session:
            last_activity = datetime.fromisoformat(session['last_activity'])
            if datetime.now() - last_activity > timedelta(minutes=15):
                session.clear()
                flash('Session expired. Please login again.', 'warning')
                return redirect(url_for('login'))
        
        session['last_activity'] = datetime.now().isoformat()
        session.permanent = True
        
        return f(*args, **kwargs)
    return decorated_function

def socketio_login_required(f):
    """Decorator to require login for socket events"""
    @wraps(f)
    def decorated_function(*args, **kwargs):
        if 'logged_in' not in session or not session['logged_in']:
            disconnect()
            return False
        
        if 'last_activity' in session:
            last_activity = datetime.fromisoformat(session['last_activity'])
            if datetime.now() - last_activity > timedelta(minutes=15):
                session.clear()
                disconnect()
                return False
        
        session['last_activity'] = datetime.now().isoformat()
        session.permanent = True
        
        return f(*args, **kwargs)
    return decorated_function

# Global variables
call_data = {
    'total_calls': 0,
    'calls': [],
    'last_update': None,
    'stats': {
        'live': 0,
        'answered': 0,
        'ringing': 0,
        'incoming': 0,
        'outgoing': 0,
        'other': 0
    },
    'dispatcher_call_stats': {}
}

dispatcher_data = {
    'total_dispatchers': 0,
    'total_sets': 0,
    'dispatchers': [],
    'sets': [],
    'last_update': None,
    'stats': {
        'active': 0,
        'inactive': 0,
        'avg_latency': 0,
        'max_latency': 0,
        'total_timeouts': 0
    }
}

# Dispatcher aliases configuration
DISPATCHER_ALIASES = {
    1: "JaipurVoda",
    2: "VijayJio", 
    3: "SurajAst",
    4: "MultyCluster"
}

data_lock = Lock()

class KamailioCallMonitor:
    def __init__(self):
        self.running = False
        self.redis_interval = 2  # Default 2 seconds for Redis
        self.dispatcher_interval = 5  # Default 5 seconds for Dispatcher
        self.redis_client = None
        self.redis_last_fetch = 0
        self.dispatcher_last_fetch = 0
        self.init_redis_connection()
        
    def init_redis_connection(self):
        """Initialize Redis connection"""
        try:
            self.redis_client = redis.Redis(
                host=REDIS_HOST,
                port=REDIS_PORT,
                decode_responses=True,
                socket_connect_timeout=5,
                socket_timeout=5
            )
            # Test connection
            self.redis_client.ping()
            logger.info(f"Redis connection established to {REDIS_HOST}:{REDIS_PORT}")
            socketio.emit('redis_connection_status', {'connected': True})
        except Exception as e:
            logger.error(f"Failed to connect to Redis: {e}")
            self.redis_client = None
            socketio.emit('redis_connection_status', {'connected': False, 'error': str(e)})
        
    def set_redis_interval(self, interval):
        """Set the Redis update interval"""
        valid_intervals = [1, 2, 3, 5, 8, 10, 15, 30, 60]
        if interval in valid_intervals:
            old_interval = self.redis_interval
            self.redis_interval = interval
            logger.info(f"Redis interval changed from {old_interval}s to {interval}s")
            
            socketio.emit('interval_updated', {
                'interval': interval, 
                'type': 'redis',
                'message': f'Call refresh rate updated to {interval} seconds'
            })
            return True
        else:
            logger.warning(f"Invalid Redis interval: {interval}. Valid options: {valid_intervals}")
            return False
    
    def set_dispatcher_interval(self, interval):
        """Set the Dispatcher update interval"""
        valid_intervals = [1, 2, 3, 5, 8, 10, 15, 30, 60]
        if interval in valid_intervals:
            old_interval = self.dispatcher_interval
            self.dispatcher_interval = interval
            logger.info(f"Dispatcher interval changed from {old_interval}s to {interval}s")
            
            socketio.emit('interval_updated', {
                'interval': interval, 
                'type': 'dispatcher',
                'message': f'Dispatcher refresh rate updated to {interval} seconds'
            })
            return True
        else:
            logger.warning(f"Invalid Dispatcher interval: {interval}. Valid options: {valid_intervals}")
            return False
    
    def decode_base64_json(self, base64_data):
        """Decode base64 encoded JSON data"""
        try:
            decoded_bytes = base64.b64decode(base64_data)
            decoded_string = decoded_bytes.decode('utf-8')
            return json.loads(decoded_string)
        except Exception as e:
            logger.error(f"Error decoding base64 JSON: {e}")
            return None
    
    def fetch_redis_call_data(self):
        """Fetch call data from Redis"""
        if not self.redis_client:
            logger.error("Redis client not available")
            return []
        
        try:
            redis_data = self.redis_client.hgetall(REDIS_KEY)
            
            if not redis_data:
                logger.debug("No data found in Redis")
                return []
            
            calls = []
            for call_id, encoded_data in redis_data.items():
                try:
                    call_value = self.decode_base64_json(encoded_data)
                    
                    if call_value:
                        call_obj = {
                            'name': call_id,
                            'value': call_value
                        }
                        calls.append(call_obj)
                        
                except Exception as e:
                    logger.error(f"Error processing call {call_id}: {e}")
                    continue
            
            logger.debug(f"Retrieved {len(calls)} calls from Redis")
            return calls
            
        except Exception as e:
            logger.error(f"Error fetching data from Redis: {e}")
            self.init_redis_connection()
            return []
    
    def calculate_running_time(self, start_time):
        """Calculate running time from start timestamp"""
        try:
            current_time = int(time.time())
            start_timestamp = int(start_time)
            duration = current_time - start_timestamp
            
            hours = duration // 3600
            minutes = (duration % 3600) // 60
            seconds = duration % 60
            
            return f"{hours:02d}:{minutes:02d}:{seconds:02d}"
        except (ValueError, TypeError):
            return "N/A"
    
    def get_call_status_class(self, status):
        """Get CSS class for call status"""
        status_lower = status.lower()
        if 'live' in status_lower or 'trying' in status_lower:
            return 'status-live'
        elif 'answered' in status_lower:
            return 'status-answered'
        elif 'ringing' in status_lower:
            return 'status-ringing'
        else:
            return 'status-other'
    
    def calculate_dispatcher_call_stats(self, calls, dispatchers):
        """Calculate call statistics by dispatcher"""
        dispatcher_stats = {}
        
        dispatcher_lookup = {}
        for dispatcher in dispatchers:
            set_id = str(dispatcher['set_id'])
            if set_id not in dispatcher_lookup:
                dispatcher_lookup[set_id] = []
            dispatcher_lookup[set_id].append(dispatcher)
        
        for call in calls:
            ibd_dispatcher = call.get('ibd_dispatcher', 'N/A')
            obd_dispatcher = call.get('obd_dispatcher', 'N/A')
            
            if ibd_dispatcher != 'N/A':
                if ibd_dispatcher not in dispatcher_stats:
                    dispatcher_stats[ibd_dispatcher] = {
                        'name': self.get_dispatcher_name(ibd_dispatcher, dispatcher_lookup),
                        'incoming': 0,
                        'outgoing': 0,
                        'total': 0
                    }
                dispatcher_stats[ibd_dispatcher]['incoming'] += 1
                dispatcher_stats[ibd_dispatcher]['total'] += 1
            
            if obd_dispatcher != 'N/A':
                if obd_dispatcher not in dispatcher_stats:
                    dispatcher_stats[obd_dispatcher] = {
                        'name': self.get_dispatcher_name(obd_dispatcher, dispatcher_lookup),
                        'incoming': 0,
                        'outgoing': 0,
                        'total': 0
                    }
                dispatcher_stats[obd_dispatcher]['outgoing'] += 1
                dispatcher_stats[obd_dispatcher]['total'] += 1
        
        return dispatcher_stats
    
    def get_dispatcher_name(self, dispatcher_id, dispatcher_lookup):
        """Get dispatcher name from lookup"""
        if '-' in str(dispatcher_id):
            parts = str(dispatcher_id).split('-')
            main_dispatcher = parts[0]
            priority = parts[1]
            
            if main_dispatcher in dispatcher_lookup and dispatcher_lookup[main_dispatcher]:
                for dispatcher in dispatcher_lookup[main_dispatcher]:
                    if str(dispatcher['priority']) == priority:
                        return f"{dispatcher['name']} (P{priority})"
                return f"{dispatcher_lookup[main_dispatcher][0]['name']} (P{priority})"
            return f"Dispatcher-{dispatcher_id}"
        else:
            if dispatcher_id in dispatcher_lookup and dispatcher_lookup[dispatcher_id]:
                for dispatcher in dispatcher_lookup[dispatcher_id]:
                    if dispatcher['priority'] == 0:
                        return dispatcher['name']
                return dispatcher_lookup[dispatcher_id][0]['name']
            try:
                return DISPATCHER_ALIASES.get(int(dispatcher_id), f"Dispatcher-{dispatcher_id}")
            except:
                return f"Dispatcher-{dispatcher_id}"
    
    def fetch_call_data(self):
        """Fetch call data from Redis and emit updates"""
        global call_data
        
        try:
            # Check Redis connection
            if not self.redis_client:
                self.init_redis_connection()
            
            if not self.redis_client:
                socketio.emit('redis_connection_status', {'connected': False, 'error': 'Redis connection failed'})
                return
            
            # Test Redis connection
            try:
                self.redis_client.ping()
                socketio.emit('redis_connection_status', {'connected': True})
            except Exception as e:
                socketio.emit('redis_connection_status', {'connected': False, 'error': 'Redis ping failed'})
                logger.error(f"Redis ping failed: {e}")
                self.init_redis_connection()
                return
            
            # Fetch data from Redis
            calls = self.fetch_redis_call_data()
            
            # Process calls - only process if we have actual data
            processed_calls = []
            stats = {
                'live': 0, 
                'answered': 0, 
                'ringing': 0, 
                'incoming': 0, 
                'outgoing': 0, 
                'other': 0
            }
            
            for call in calls:
                value = call.get('value', {})
                
                if not value or len(value) == 0:
                    continue
                    
                status = value.get('Call status', 'Unknown')
                start_time = value.get('Start Time', '')
                call_type = value.get('Call Type', 'Unknown')
                call_id = value.get('CALL ID', call.get('name', ''))
                
                if (not status or 
                    status == 'Unknown' or 
                    not call_id or 
                    call_id.strip() == ''):
                    continue
                
                processed_call = {
                    'id': call_id,
                    'status': status,
                    'status_class': self.get_call_status_class(status),
                    'call_type': call_type,
                    'ibd_dispatcher': value.get('Incoming Dispatcher', 'N/A'),
                    'obd_dispatcher': value.get('Outgoing Dispatcher', 'N/A'),
                    'running_time': self.calculate_running_time(start_time),
                    'start_time': start_time,
                    'caller': value.get('Source User', 'N/A'),
                    'callee': value.get('Destination User', 'N/A'),
                    'raw_value': value
                }
                
                processed_calls.append(processed_call)
                
                # Update stats
                status_lower = status.lower()
                if 'live' in status_lower or 'trying' in status_lower:
                    stats['live'] += 1
                elif 'answered' in status_lower:
                    stats['answered'] += 1
                elif 'ringing' in status_lower:
                    stats['ringing'] += 1
                else:
                    stats['other'] += 1
                
                call_type_lower = call_type.lower()
                if 'incoming' in call_type_lower or 'ibd' in call_type_lower:
                    stats['incoming'] += 1
                elif 'outgoing' in call_type_lower or 'obd' in call_type_lower:
                    stats['outgoing'] += 1
            
            # Calculate dispatcher call statistics
            dispatcher_call_stats = self.calculate_dispatcher_call_stats(processed_calls, dispatcher_data.get('dispatchers', []))
            
            # Update global data
            final_call_data = {
                'total_calls': len(processed_calls),
                'calls': processed_calls,
                'last_update': datetime.now().isoformat(),
                'stats': stats,
                'dispatcher_call_stats': dispatcher_call_stats,
                'redis_interval': self.redis_interval,
                'dispatcher_interval': self.dispatcher_interval
            }
            
            with data_lock:
                call_data.update(final_call_data)
            
            # Emit call data update to all connected clients
            socketio.emit('call_data_update', final_call_data)
            logger.debug(f"Emitted call data update: {len(processed_calls)} calls")
            
        except Exception as e:
            logger.error(f"Error in fetch_call_data: {e}")
            socketio.emit('redis_connection_status', {'connected': False, 'error': str(e)})
    
    def extract_name_from_attrs(self, attrs_body):
        """Extract Name from ATTRS BODY string"""
        if not attrs_body:
            return None
            
        name_match = re.search(r'Name=([^;]+)', attrs_body)
        if name_match:
            return name_match.group(1).strip()
        return None
    
    def parse_dispatcher_output(self, output):
        """Parse kamcmd dispatcher.list output"""
        dispatchers = []
        sets = []
        
        if not output or output.strip() == "":
            return dispatchers, sets
            
        try:
            lines = output.strip().split('\n')
            
            in_records = False
            current_set = None
            current_dest = None
            current_attrs = None
            current_latency = None
            
            for line in lines:
                stripped = line.strip()
                if not stripped:
                    continue
                    
                if stripped == 'RECORDS: {':
                    in_records = True
                    continue
                elif not in_records:
                    continue
                    
                if stripped == 'SET: {':
                    current_set = {}
                    current_dest = None
                    current_attrs = None
                    current_latency = None
                elif stripped.startswith('ID: '):
                    if current_set is not None:
                        current_set['ID'] = int(stripped.split(': ')[1])
                elif stripped == 'TARGETS: {':
                    current_set['TARGETS'] = []
                elif stripped == 'DEST: {':
                    current_dest = {}
                elif stripped == 'ATTRS: {':
                    current_attrs = {}
                elif stripped == 'LATENCY: {':
                    current_latency = {}
                elif stripped == '}':
                    if current_latency is not None and current_dest is not None and 'LATENCY' not in current_dest:
                        current_dest['LATENCY'] = current_latency
                        current_latency = None
                    elif current_attrs is not None and current_dest is not None and 'ATTRS' not in current_dest:
                        current_dest['ATTRS'] = current_attrs
                        current_attrs = None
                    elif current_dest is not None and current_set is not None:
                        current_set['TARGETS'].append(current_dest)
                        current_dest = None
                    elif current_set is not None:
                        self.process_dispatcher_set(current_set, dispatchers, sets)
                        current_set = None
                elif ': ' in stripped:
                    key, value = stripped.split(': ', 1)
                    
                    if current_latency is not None:
                        try:
                            current_latency[key] = float(value) if '.' in value else int(value)
                        except ValueError:
                            current_latency[key] = value
                    elif current_attrs is not None:
                        current_attrs[key] = value.replace('<null string>', '')
                    elif current_dest is not None:
                        if key in ['PRIORITY']:
                            current_dest[key] = int(value)
                        else:
                            current_dest[key] = value
                    elif current_set is not None:
                        current_set[key] = value
                        
        except Exception as e:
            logger.error(f"Error parsing dispatcher output: {e}")
            
        return dispatchers, sets
    
    def process_dispatcher_set(self, set_data, dispatchers, sets):
        """Process a complete dispatcher set"""
        try:
            set_id = int(set_data.get('ID', 0))
            set_alias = DISPATCHER_ALIASES.get(set_id, f"Set-{set_id}")
            
            processed_set = {
                'id': set_id,
                'alias': set_alias,
                'targets': [],
                'total_targets': 0,
                'active_targets': 0,
                'avg_latency': 0,
                'max_latency': 0,
                'total_timeouts': 0
            }
            
            targets = set_data.get('TARGETS', [])
            if not isinstance(targets, list):
                targets = [targets] if targets else []
                
            latencies = []
            total_timeouts = 0
            
            for dest in targets:
                if not isinstance(dest, dict):
                    continue
                    
                uri = dest.get('URI', '')
                flags = dest.get('FLAGS', '')
                priority = int(dest.get('PRIORITY', 0))
                latency_info = dest.get('LATENCY', {})
                attrs = dest.get('ATTRS', {})
                
                ip_match = re.search(r'sip:([^:]+)', uri)
                ip = ip_match.group(1) if ip_match else uri.replace('sip:', '').split(':')[0]
                
                attrs_body = attrs.get('BODY', '')
                dispatcher_name = self.extract_name_from_attrs(attrs_body)
                if not dispatcher_name:
                    if set_id == 4:
                        dispatcher_name = "MultyCluster"
                    else:
                        dispatcher_name = f"{set_alias}-{ip}"
                
                avg_latency = float(latency_info.get('AVG', 0))
                max_latency = float(latency_info.get('MAX', 0))
                timeouts = int(latency_info.get('TIMEOUT', 0))
                
                status, is_active = self.get_status_from_flags(flags)
                
                dispatcher = {
                    'set_id': set_id,
                    'set_alias': set_alias,
                    'name': dispatcher_name,
                    'uri': uri,
                    'ip': ip,
                    'flags': flags,
                    'priority': priority,
                    'status': status,
                    'is_active': is_active,
                    'latency': {
                        'avg': avg_latency,
                        'std': float(latency_info.get('STD', 0)),
                        'est': float(latency_info.get('EST', 0)),
                        'max': max_latency,
                        'current': float(latency_info.get('EST', 0)),
                        'timeout': timeouts
                    }
                }
                
                dispatchers.append(dispatcher)
                processed_set['targets'].append(dispatcher)
                
                if is_active:
                    processed_set['active_targets'] += 1
                    if avg_latency > 0:
                        latencies.append(avg_latency)
                        
                total_timeouts += timeouts
                
            processed_set['total_targets'] = len(targets)
            processed_set['avg_latency'] = sum(latencies) / len(latencies) if latencies else 0
            processed_set['max_latency'] = max([d['latency']['max'] for d in processed_set['targets']]) if processed_set['targets'] else 0
            processed_set['total_timeouts'] = total_timeouts
            
            sets.append(processed_set)
            
        except Exception as e:
            logger.error(f"Error processing dispatcher set: {e}")
    
    def get_status_from_flags(self, flags):
        """Get status and active state from flags"""
        flags = flags.upper()
        
        if flags == 'AP':
            return 'Active', True
        elif flags == 'AX':
            return 'Checking', True
        elif flags == 'IP':
            return 'Inactive', False
        elif flags == 'IX':
            return 'Checking', False
        elif flags == 'DX':
            return 'Disabled', False
        elif flags == 'DP':
            return 'Disabled', False
        else:
            if 'A' in flags:
                return 'Active', True
            elif 'I' in flags:
                return 'Inactive', False
            elif 'D' in flags:
                return 'Disabled', False
            else:
                return 'Unknown', False
    
    def fetch_dispatcher_data(self):
        """Fetch dispatcher data from kamcmd and emit updates"""
        global dispatcher_data
        
        try:
            result = subprocess.run(
                ['kamcmd', 'dispatcher.list'],
                capture_output=True,
                text=True,
                timeout=30
            )
            
            if result.returncode != 0:
                logger.error(f"kamcmd failed with return code {result.returncode}")
                logger.error(f"STDERR: {result.stderr}")
                socketio.emit('dispatcher_connection_status', {'connected': False, 'error': 'dispatcher.list failed'})
                return
            
            output = result.stdout.strip()
            if not output:
                logger.warning("No output from kamcmd dispatcher.list")
                socketio.emit('dispatcher_connection_status', {'connected': False, 'error': 'No dispatcher response'})
                return
            
            socketio.emit('dispatcher_connection_status', {'connected': True})
            
            dispatchers, sets = self.parse_dispatcher_output(output)
            
            if not dispatchers:
                logger.warning("No dispatchers parsed from kamcmd output")
                # Still emit empty data to show system is connected but no dispatchers configured
                final_dispatcher_data = {
                    'total_dispatchers': 0,
                    'total_sets': 0,
                    'dispatchers': [],
                    'sets': [],
                    'last_update': datetime.now().isoformat(),
                    'stats': {
                        'active': 0,
                        'inactive': 0,
                        'avg_latency': 0,
                        'max_latency': 0,
                        'total_timeouts': 0
                    }
                }
            else:
                # Calculate overall stats
                total_dispatchers = len(dispatchers)
                active_dispatchers = len([d for d in dispatchers if d['is_active']])
                inactive_dispatchers = total_dispatchers - active_dispatchers
                
                active_latencies = [d['latency']['avg'] for d in dispatchers if d['is_active'] and d['latency']['avg'] > 0]
                avg_latency = sum(active_latencies) / len(active_latencies) if active_latencies else 0
                
                all_latencies = [d['latency']['max'] for d in dispatchers if d['latency']['max'] > 0]
                max_latency = max(all_latencies) if all_latencies else 0
                
                total_timeouts = sum([d['latency']['timeout'] for d in dispatchers])
                
                final_dispatcher_data = {
                    'total_dispatchers': total_dispatchers,
                    'total_sets': len(sets),
                    'dispatchers': dispatchers,
                    'sets': sets,
                    'last_update': datetime.now().isoformat(),
                    'stats': {
                        'active': active_dispatchers,
                        'inactive': inactive_dispatchers,
                        'avg_latency': round(avg_latency, 2),
                        'max_latency': max_latency,
                        'total_timeouts': total_timeouts
                    }
                }
            
            with data_lock:
                dispatcher_data.update(final_dispatcher_data)
            
            socketio.emit('dispatcher_data_update', final_dispatcher_data)
            logger.debug(f"Emitted dispatcher data update: {len(dispatchers)} dispatchers")
            
        except subprocess.TimeoutExpired:
            logger.error("kamcmd command timeout")
            socketio.emit('dispatcher_connection_status', {'connected': False, 'error': 'Command timeout'})
        except FileNotFoundError:
            logger.error("kamcmd not found in PATH")
            socketio.emit('dispatcher_connection_status', {'connected': False, 'error': 'kamcmd not found'})
        except Exception as e:
            logger.error(f"Error in fetch_dispatcher_data: {e}")
            socketio.emit('dispatcher_connection_status', {'connected': False, 'error': str(e)})
    
    def start_monitoring(self):
        """Start monitoring in background thread with separate intervals"""
        self.running = True
        last_redis_fetch = 0
        last_dispatcher_fetch = 0
        
        logger.info(f"Starting monitoring with Redis interval: {self.redis_interval}s, Dispatcher interval: {self.dispatcher_interval}s")
        
        while self.running:
            current_time = time.time()
            
            # Check if it's time to fetch Redis data
            if current_time - last_redis_fetch >= self.redis_interval:
                logger.debug(f"Fetching Redis data (interval: {self.redis_interval}s)")
                self.fetch_call_data()
                last_redis_fetch = current_time
            
            # Check if it's time to fetch Dispatcher data
            if current_time - last_dispatcher_fetch >= self.dispatcher_interval:
                logger.debug(f"Fetching Dispatcher data (interval: {self.dispatcher_interval}s)")
                self.fetch_dispatcher_data()
                last_dispatcher_fetch = current_time
            
            time.sleep(0.5)
    
    def stop_monitoring(self):
        """Stop monitoring"""
        self.running = False
        logger.info("Monitoring stopped")

# Initialize monitor
monitor = KamailioCallMonitor()

@app.route('/login', methods=['GET', 'POST'])
def login():
    """Login page"""
    if request.method == 'POST':
        username = request.form.get('username')
        password = request.form.get('password')
        
        if username == USERNAME and check_password_hash(PASSWORD_HASH, password):
            session['logged_in'] = True
            session['username'] = username
            session['last_activity'] = datetime.now().isoformat()
            session.permanent = True
            flash('Login successful!', 'success')
            return redirect(url_for('overview'))
        else:
            flash('Invalid username or password!', 'error')
    
    return render_template('login.html')

@app.route('/logout')
def logout():
    """Logout and clear session"""
    session.clear()
    flash('You have been logged out.', 'info')
    return redirect(url_for('login'))

@app.route('/')
@login_required
def index():
    """Redirect to overview page"""
    return redirect(url_for('overview'))

@app.route('/overview')
@login_required
def overview():
    """Overview dashboard page"""
    return render_template('overview.html')

@app.route('/calls')
@login_required
def calls():
    """Live Calls page"""
    return render_template('calls.html')

@app.route('/dispatchers')
@login_required
def dispatchers():
    """Dispatchers page"""
    return render_template('dispatchers.html')

@app.route('/api/calls')
@login_required
def get_calls():
    """API endpoint to get call data"""
    with data_lock:
        return jsonify(call_data)

@app.route('/api/dispatchers')
@login_required
def get_dispatchers():
    """API endpoint to get dispatcher data"""
    with data_lock:
        return jsonify(dispatcher_data)

@app.route('/api/stats')
@login_required
def get_stats():
    """API endpoint to get statistics only"""
    with data_lock:
        return jsonify({
            'total_calls': call_data['total_calls'],
            'stats': call_data['stats'],
            'dispatcher_call_stats': call_data.get('dispatcher_call_stats', {}),
            'last_update': call_data['last_update'],
            'dispatcher_stats': dispatcher_data['stats'],
            'dispatcher_last_update': dispatcher_data['last_update']
        })

@app.route('/api/set_redis_interval/<int:interval>')
@login_required
def set_redis_interval(interval):
    """API endpoint to set Redis refresh interval"""
    if monitor.set_redis_interval(interval):
        return jsonify({'success': True, 'interval': interval, 'type': 'redis'})
    return jsonify({'success': False, 'error': 'Invalid interval'})

@app.route('/api/set_dispatcher_interval/<int:interval>')
@login_required
def set_dispatcher_interval(interval):
    """API endpoint to set Dispatcher refresh interval"""
    if monitor.set_dispatcher_interval(interval):
        return jsonify({'success': True, 'interval': interval, 'type': 'dispatcher'})
    return jsonify({'success': False, 'error': 'Invalid interval'})

@app.route('/api/session_check')
@login_required
def session_check():
    """Check if session is still valid"""
    return jsonify({'valid': True, 'username': session.get('username')})

@socketio.on('connect')
@socketio_login_required
def handle_connect():
    """Handle client connection"""
    logger.info(f"Client connected")
    with data_lock:
        emit('call_data_update', call_data)
        emit('dispatcher_data_update', dispatcher_data)
        emit('interval_status', {
            'redis_interval': monitor.redis_interval,
            'dispatcher_interval': monitor.dispatcher_interval
        })

@socketio.on('disconnect')
def handle_disconnect():
    """Handle client disconnection"""
    logger.info(f"Client disconnected")

@socketio.on('request_update')
@socketio_login_required
def handle_request_update():
    """Handle manual update request"""
    logger.info("Manual update requested")
    
    # Force immediate fetch
    monitor.fetch_call_data()
    monitor.fetch_dispatcher_data()
    
    with data_lock:
        emit('call_data_update', call_data)
        emit('dispatcher_data_update', dispatcher_data)

@socketio.on('set_redis_interval')
@socketio_login_required
def handle_set_redis_interval(data):
    """Handle Redis refresh interval change"""
    interval = data.get('interval', 2)
    logger.info(f"Redis interval change requested: {interval}s")
    
    if monitor.set_redis_interval(interval):
        emit('interval_updated', {'interval': interval, 'type': 'redis'})
    else:
        emit('error', {'message': f'Invalid Redis interval: {interval}'})

@socketio.on('set_dispatcher_interval')
@socketio_login_required
def handle_set_dispatcher_interval(data):
    """Handle Dispatcher refresh interval change"""
    interval = data.get('interval', 5)
    logger.info(f"Dispatcher interval change requested: {interval}s")
    
    if monitor.set_dispatcher_interval(interval):
        emit('interval_updated', {'interval': interval, 'type': 'dispatcher'})
    else:
        emit('error', {'message': f'Invalid Dispatcher interval: {interval}'})

def start_background_monitoring():
    """Start background monitoring thread"""
    monitor_thread = Thread(target=monitor.start_monitoring, daemon=True)
    monitor_thread.start()
    logger.info("Background monitoring thread started")

# Add these utility functions
def get_db_connection():
    """Get MySQL database connection"""
    try:
        connection = pymysql.connect(**MYSQL_CONFIG)
        return connection
    except Exception as e:
        logger.error(f"Database connection error: {e}")
        return None

def inet_aton(ip):
    """Convert IP address to integer"""
    try:
        return struct.unpack("!I", socket.inet_aton(ip))[0]
    except:
        return 0

# Add this new route for configuration page
@app.route('/config')
@login_required
def config():
    """Configuration page"""
    return render_template('config.html')

# Add all these configuration API routes
@app.route('/api/config/allowed_ips')
@login_required
def get_allowed_ips():
    """Get allowed IPs"""
    search = request.args.get('search', '')
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor(pymysql.cursors.DictCursor)
        if search:
            query = "SELECT * FROM allow_ip WHERE ip_addr LIKE %s OR reason LIKE %s ORDER BY added_at DESC"
            cursor.execute(query, (f'%{search}%', f'%{search}%'))
        else:
            query = "SELECT * FROM allow_ip ORDER BY added_at DESC"
            cursor.execute(query)

        results = cursor.fetchall()
        return jsonify({'success': True, 'data': results})
    except Exception as e:
        logger.error(f"Error fetching allowed IPs: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/allowed_ips', methods=['POST'])
@login_required
def add_allowed_ip():
    """Add allowed IP"""
    data = request.get_json()
    ip_addr = data.get('ip_addr')
    reason = data.get('reason')

    if not ip_addr or not reason:
        return jsonify({'error': 'IP address and reason are required'}), 400

    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "INSERT INTO allow_ip (ip_addr, reason, added_at) VALUES (%s, %s, NOW())"
        cursor.execute(query, (ip_addr, reason))
        conn.commit()
        return jsonify({'success': True, 'message': 'IP added successfully'})
    except Exception as e:
        logger.error(f"Error adding allowed IP: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/allowed_ips/<int:id>', methods=['DELETE'])
#@app.route('/api/config/allowed_ips/undefined' methods=['DELETE'])
@login_required
def delete_allowed_ip(id):
    """Delete allowed IP"""
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "DELETE FROM allow_ip WHERE id = %s"
        cursor.execute(query, (id,))
        conn.commit()
        return jsonify({'success': True, 'message': 'IP deleted successfully'})
    except Exception as e:
        logger.error(f"Error deleting allowed IP: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/blacklist_numbers')
@login_required
def get_blacklist_numbers():
    """Get blacklist numbers"""
    search = request.args.get('search', '')
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor(pymysql.cursors.DictCursor)
        if search:
            query = "SELECT * FROM blacklist_numbers WHERE number LIKE %s OR reason LIKE %s ORDER BY id DESC"
            cursor.execute(query, (f'%{search}%', f'%{search}%'))
        else:
            query = "SELECT * FROM blacklist_numbers ORDER BY id DESC"
            cursor.execute(query)

        results = cursor.fetchall()
        return jsonify({'success': True, 'data': results})
    except Exception as e:
        logger.error(f"Error fetching blacklist numbers: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/blacklist_numbers', methods=['POST'])
@login_required
def add_blacklist_number():
    """Add blacklist number"""
    data = request.get_json()
    number = data.get('number')
    reason = data.get('reason')

    if not number or not reason:
        return jsonify({'error': 'Number and reason are required'}), 400

    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "INSERT INTO blacklist_numbers (number, reason) VALUES (%s, %s)"
        cursor.execute(query, (number, reason))
        conn.commit()
        return jsonify({'success': True, 'message': 'Number added to blacklist successfully'})
    except Exception as e:
        logger.error(f"Error adding blacklist number: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/blacklist_numbers/<int:id>', methods=['DELETE'])
@login_required
def delete_blacklist_number(id):
    """Delete blacklist number"""
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "DELETE FROM blacklist_numbers WHERE id = %s"
        cursor.execute(query, (id,))
        conn.commit()
        return jsonify({'success': True, 'message': 'Number removed from blacklist successfully'})
    except Exception as e:
        logger.error(f"Error deleting blacklist number: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/calls_on_block')
@login_required
def get_calls_on_block():
    """Get blocked calls report"""
    search = request.args.get('search', '')
    date_from = request.args.get('date_from', '')
    date_to = request.args.get('date_to', '')

    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor(pymysql.cursors.DictCursor)
        query = "SELECT * FROM CALLSONBLOCK WHERE 1=1"
        params = []

        if search:
            query += " AND (fromnumber LIKE %s OR tonumber LIKE %s OR sendcode LIKE %s)"
            params.extend([f'%{search}%', f'%{search}%', f'%{search}%'])

        if date_from:
            query += " AND added_at >= %s"
            params.append(date_from)

        if date_to:
            query += " AND added_at <= %s"
            params.append(date_to)

        query += " ORDER BY added_at DESC LIMIT 1000"
        cursor.execute(query, params)

        results = cursor.fetchall()
        return jsonify({'success': True, 'data': results})
    except Exception as e:
        logger.error(f"Error fetching blocked calls: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/dispatchers')
@login_required
def get_dispatchers_config():
    """Get dispatcher configuration"""
    search = request.args.get('search', '')
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor(pymysql.cursors.DictCursor)
        if search:
            query = "SELECT * FROM dispatcher WHERE setid LIKE %s OR destination LIKE %s OR description LIKE %s ORDER BY setid, priority"
            cursor.execute(query, (f'%{search}%', f'%{search}%', f'%{search}%'))
        else:
            query = "SELECT * FROM dispatcher ORDER BY setid, priority"
            cursor.execute(query)

        results = cursor.fetchall()

        # Parse attrs field for each result
        for result in results:
            if result['attrs']:
                # Parse attrs to extract socket, from_uri, and Name
                attrs = result['attrs']
                socket_match = re.search(r'socket=([^;]+)', attrs)
                from_uri_match = re.search(r'from_uri=([^;]+)', attrs)
                name_match = re.search(r'Name=([^;]+)', attrs)

                result['socket'] = socket_match.group(1) if socket_match else ''
                result['from_uri'] = from_uri_match.group(1) if from_uri_match else ''
                result['name'] = name_match.group(1) if name_match else ''
            else:
                result['socket'] = ''
                result['from_uri'] = ''
                result['name'] = ''

        return jsonify({'success': True, 'data': results})
    except Exception as e:
        logger.error(f"Error fetching dispatchers: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/dispatchers', methods=['POST'])
@login_required
def add_dispatcher():
    """Add dispatcher"""
    data = request.get_json()
    setid = data.get('setid')
    destination = data.get('destination')
    flags = data.get('flags', 0)
    priority = data.get('priority', 0)
    description = data.get('description', '')
    socket = data.get('socket', '')
    from_uri = data.get('from_uri', '')
    name = data.get('name', '')

    if not setid or not destination:
        return jsonify({'error': 'Set ID and destination are required'}), 400

    # Build attrs string
    attrs_parts = []
    if socket:
        attrs_parts.append(f"socket={socket}")
    if from_uri:
        attrs_parts.append(f"from_uri={from_uri}")
    if name:
        attrs_parts.append(f"Name={name}")

    attrs = ';'.join(attrs_parts)

    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "INSERT INTO dispatcher (setid, destination, flags, priority, attrs, description) VALUES (%s, %s, %s, %s, %s, %s)"
        cursor.execute(query, (setid, destination, flags, priority, attrs, description))
        conn.commit()
        return jsonify({'success': True, 'message': 'Dispatcher added successfully'})
    except Exception as e:
        logger.error(f"Error adding dispatcher: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/dispatchers/<int:id>', methods=['PUT'])
@login_required
def update_dispatcher(id):
    """Update dispatcher"""
    data = request.get_json()
    setid = data.get('setid')
    destination = data.get('destination')
    flags = data.get('flags', 0)
    priority = data.get('priority', 0)
    description = data.get('description', '')
    socket = data.get('socket', '')
    from_uri = data.get('from_uri', '')
    name = data.get('name', '')

    if not setid or not destination:
        return jsonify({'error': 'Set ID and destination are required'}), 400

    # Build attrs string
    attrs_parts = []
    if socket:
        attrs_parts.append(f"socket={socket}")
    if from_uri:
        attrs_parts.append(f"from_uri={from_uri}")
    if name:
        attrs_parts.append(f"Name={name}")

    attrs = ';'.join(attrs_parts)

    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "UPDATE dispatcher SET setid=%s, destination=%s, flags=%s, priority=%s, attrs=%s, description=%s WHERE id=%s"
        cursor.execute(query, (setid, destination, flags, priority, attrs, description, id))
        conn.commit()
        return jsonify({'success': True, 'message': 'Dispatcher updated successfully'})
    except Exception as e:
        logger.error(f"Error updating dispatcher: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/dispatchers/<int:id>', methods=['DELETE'])
@login_required
def delete_dispatcher(id):
    """Delete dispatcher"""
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "DELETE FROM dispatcher WHERE id = %s"
        cursor.execute(query, (id,))
        conn.commit()
        return jsonify({'success': True, 'message': 'Dispatcher deleted successfully'})
    except Exception as e:
        logger.error(f"Error deleting dispatcher: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/dispatcher_reload', methods=['POST'])
@login_required
def reload_dispatcher():
    """Reload dispatcher using kamcmd"""
    try:
        result = subprocess.run(
            ['kamcmd', 'dispatcher.reload'],
            capture_output=True,
            text=True,
            timeout=30
        )

        if result.returncode == 0:
            return jsonify({'success': True, 'message': 'Dispatcher reloaded successfully'})
        else:
            return jsonify({'error': f'Dispatcher reload failed: {result.stderr}'}), 500
    except subprocess.TimeoutExpired:
        return jsonify({'error': 'Dispatcher reload timeout'}), 500
    except FileNotFoundError:
        return jsonify({'error': 'kamcmd command not found'}), 500
    except Exception as e:
        logger.error(f"Error reloading dispatcher: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/config/cli_routes')
@login_required
def get_cli_routes():
    """Get CLI routes"""
    search = request.args.get('search', '')
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor(pymysql.cursors.DictCursor)
        if search:
            query = """SELECT * FROM CALLERIDWITHDIS
                      WHERE clistart LIKE %s OR cliend LIKE %s OR dispatcherid LIKE %s
                      OR fromsockt LIKE %s OR prefix LIKE %s
                      ORDER BY added_at DESC"""
            search_param = f'%{search}%'
            cursor.execute(query, (search_param, search_param, search_param, search_param, search_param))
        else:
            query = "SELECT * FROM CALLERIDWITHDIS ORDER BY added_at DESC"
            cursor.execute(query)

        results = cursor.fetchall()
        return jsonify({'success': True, 'data': results})
    except Exception as e:
        logger.error(f"Error fetching CLI routes: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/cli_routes', methods=['POST'])
@login_required
def add_cli_route():
    """Add CLI route"""
    data = request.get_json()
    clistart = data.get('clistart')
    cliend = data.get('cliend')
    dispatcherid = data.get('dispatcherid')
    dispatcherpriority = data.get('dispatcherpriority')
    fromsockt = data.get('fromsockt')
    prefix = data.get('prefix')

    if not all([clistart, cliend, dispatcherid, dispatcherpriority, fromsockt, prefix]):
        return jsonify({'error': 'All fields are required'}), 400

    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = """INSERT INTO CALLERIDWITHDIS
                   (clistart, cliend, dispatcherid, dispatcherpriority, fromsockt, prefix, added_at)
                   VALUES (%s, %s, %s, %s, %s, %s, NOW())"""
        cursor.execute(query, (clistart, cliend, dispatcherid, dispatcherpriority, fromsockt, prefix))
        conn.commit()
        return jsonify({'success': True, 'message': 'CLI route added successfully'})
    except Exception as e:
        logger.error(f"Error adding CLI route: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/cli_routes/<int:id>', methods=['DELETE'])
@login_required
def delete_cli_route(id):
    """Delete CLI route"""
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "DELETE FROM CALLERIDWITHDIS WHERE id = %s"
        cursor.execute(query, (id,))
        conn.commit()
        return jsonify({'success': True, 'message': 'CLI route deleted successfully'})
    except Exception as e:
        logger.error(f"Error deleting CLI route: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/ip_ranges')
@login_required
def get_ip_ranges():
    """Get IP ranges for RTP"""
    search = request.args.get('search', '')
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor(pymysql.cursors.DictCursor)
        if search:
            query = """SELECT start_ip, end_ip, rtpname, added_at FROM ip_ranges
                      WHERE start_ip LIKE %s OR end_ip LIKE %s OR rtpname LIKE %s
                      ORDER BY added_at DESC"""
            search_param = f'%{search}%'
            cursor.execute(query, (search_param, search_param, search_param))
        else:
            query = "SELECT start_ip, end_ip, rtpname, added_at FROM ip_ranges ORDER BY added_at DESC"
            cursor.execute(query)

        results = cursor.fetchall()
        return jsonify({'success': True, 'data': results})
    except Exception as e:
        logger.error(f"Error fetching IP ranges: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/ip_ranges', methods=['POST'])
@login_required
def add_ip_range():
    """Add IP range"""
    data = request.get_json()
    start_ip = data.get('start_ip')
    end_ip = data.get('end_ip')
    rtpname = data.get('rtpname')

    if not all([start_ip, end_ip, rtpname]):
        return jsonify({'error': 'Start IP, End IP, and RTP Name are required'}), 400

    # Convert IPs to integers
    start_ip_int = inet_aton(start_ip)
    end_ip_int = inet_aton(end_ip)

    if start_ip_int == 0 or end_ip_int == 0:
        return jsonify({'error': 'Invalid IP address format'}), 400

    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = """INSERT INTO ip_ranges
                   (start_ip, end_ip, start_ip_int, end_ip_int, rtpname, added_at)
                   VALUES (%s, %s, %s, %s, %s, NOW())"""
        cursor.execute(query, (start_ip, end_ip, start_ip_int, end_ip_int, rtpname))
        conn.commit()
        return jsonify({'success': True, 'message': 'IP range added successfully'})
    except Exception as e:
        logger.error(f"Error adding IP range: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

@app.route('/api/config/ip_ranges/<int:id>', methods=['DELETE'])
@login_required
def delete_ip_range(id):
    """Delete IP range"""
    conn = get_db_connection()
    if not conn:
        return jsonify({'error': 'Database connection failed'}), 500

    try:
        cursor = conn.cursor()
        query = "DELETE FROM ip_ranges WHERE id = %s"
        cursor.execute(query, (id,))
        conn.commit()
        return jsonify({'success': True, 'message': 'IP range deleted successfully'})
    except Exception as e:
        logger.error(f"Error deleting IP range: {e}")
        return jsonify({'error': str(e)}), 500
    finally:
        conn.close()

if __name__ == '__main__':
    logger.info("Starting Kamailio Live Calls Monitor - Production Version")
    logger.info(f"Redis: {REDIS_HOST}:{REDIS_PORT}, Key: {REDIS_KEY}")
    
    # Create templates directory if it doesn't exist
    templates_dir = os.path.join(os.path.dirname(__file__), 'templates')
    os.makedirs(templates_dir, exist_ok=True)
    
    # Start background monitoring
    start_background_monitoring()
    
    # Run Flask app
    try:
        socketio.run(app, host='0.0.0.0', port=80, debug=False, allow_unsafe_werkzeug=True)
    except KeyboardInterrupt:
        logger.info("Shutting down...")
        monitor.stop_monitoring()
